核心验证器（Core Validators）
===============

Yii 提供一系列常用的核心验证器，主要存在于 `yii\validators` 命名空间之下。为了避免使用冗长的类名，你可以直接用**昵称**来指定相应的核心验证器。比如你可以用 `required` 昵称代指 [[yii\validators\RequiredValidator]] 类：

```php
public function rules()
{
    return [
        [['email', 'password'], 'required'],
    ];
}
```

[[yii\validators\Validator::builtInValidators]] 属性声明了所有被支持的验证器昵称。

下面，我们将详细介绍每一款验证器的主要用法和属性。


## [[yii\validators\BooleanValidator|boolean（布尔型）]] <a name="boolean"></a>

```php
[
    // 检查 "selected" 是否为 0 或 1，无视数据类型
    ['selected', 'boolean'],

    // 检查 "deleted" 是否为布尔类型，即 true 或 false
    ['deleted', 'boolean', 'trueValue' => true, 'falseValue' => false, 'strict' => true],
]
```

该验证器检查输入值是否为一个布尔值。

- `trueValue`： 代表**真**的值。默认为 `'1'`。
- `falseValue`：代表**假**的值。默认为 `'0'`。
- `strict`：是否要求待测输入必须严格匹配 `trueValue` 或 `falseValue`。默认为 `false`。


> 注意：因为通过 HTML 表单传递的输入数据都是字符串类型，所以一般情况下你都需要保持
  [[yii\validators\BooleanValidator::strict|strict]] 属性为假。


## [[yii\captcha\CaptchaValidator|captcha（验证码）]] <a name="captcha"></a>

```php
[
    ['verificationCode', 'captcha'],
]
```

该验证器通常配合 [[yii\captcha\CaptchaAction]] 以及 [[yii\captcha\Captcha]]
使用，以确保某一输入与 [[yii\captcha\Captcha|CAPTCHA]] 小部件所显示的验证代码（verification code）相同。

- `caseSensitive`：对验证代码的比对是否要求大小写敏感。默认为 false。
- `captchaAction`：指向用于渲染 CAPTCHA 图片的 [[yii\captcha\CaptchaAction|CAPTCHA action]] 的 [路由](structure-controllers.md#routes)。默认为 `'site/captcha'`。
- `skipOnEmpty`：当输入为空时，是否跳过验证。默认为 false，也就是输入值为必需项。
  

## [[yii\validators\CompareValidator|compare（比较）]] <a name="compare"></a>

```php
[
    // 检查 "password" 特性的值是否与 "password_repeat" 的值相同
    ['password', 'compare'],

    // 检查年龄是否大于等于 30
    ['age', 'compare', 'compareValue' => 30, 'operator' => '>='],
]
```

该验证器比较两个特定输入值之间的关系是否与 `operator` 属性所指定的相同。

- `compareAttribute`：用于与原特性相比较的特性名称。当该验证器被用于验证某目标特性时，该属性会默认为目标属性加后缀 `_repeat`。举例来说，若目标特性为 `password`，则该属性默认为 `password_repeat`。
- `compareValue`：用于与输入值相比较的常量值。当该属性与 `compareAttribute` 属性同时被指定时，该属性优先被使用。
- `operator`：比较操作符。默认为 `==`，意味着检查输入值是否与 `compareAttribute` 或 `compareValue` 的值相等。该属性支持如下操作符：
     * `==`：检查两值是否相等。比对为非严格模式。
     * `===`：检查两值是否全等。比对为严格模式。
     * `!=`：检查两值是否不等。比对为非严格模式。
     * `!==`：检查两值是否不全等。比对为严格模式。
     * `>`：检查待测目标值是否大于给定被测值。
     * `>=`：检查待测目标值是否大于等于给定被测值。
     * `<`：检查待测目标值是否小于给定被测值。
     * `<=`：检查待测目标值是否小于等于给定被测值。


## [[yii\validators\DateValidator|date（日期）]] <a name="date"></a>

```php
[
    [['from', 'to'], 'date'],
]
```

该验证器检查输入值是否为适当格式的 date，time，或者 datetime。另外，它还可以帮你把输入值转换为一个 UNIX 时间戳并保存到 [[yii\validators\DateValidator::timestampAttribute|timestampAttribute]] 属性所指定的特性里。

- `format`：待测的 日期/时间 格式。请参考
  [date_create_from_format() 相关的 PHP 手册](http://www.php.net/manual/zh/datetime.createfromformat.php)了解设定格式字符串的更多细节。默认值为 `'Y-m-d'`。
- `timestampAttribute`：用于保存用输入时间/日期转换出来的 UNIX 时间戳的特性。


## [[yii\validators\DefaultValueValidator|default（默认值）]] <a name="default"></a>

```php
[
    // 若 "age" 为空，则将其设为 null
    ['age', 'default', 'value' => null],

    // 若 "country" 为空，则将其设为 "USA"
    ['country', 'default', 'value' => 'USA'],

    // 若 "from" 和 "to" 为空，则分别给他们分配自今天起，3 天后和 6 天后的日期。
    [['from', 'to'], 'default', 'value' => function ($model, $attribute) {
        return date('Y-m-d', strtotime($attribute === 'to' ? '+3 days' ：'+6 days'));
    }],
]
```

该验证器并不进行数据验证。而是，给为空的待测特性分配默认值。

- `value`：默认值，或一个返回默认值的 PHP Callable 对象（即回调函数）。它们会分配给检测为空的待测特性。PHP 回调方法的样式如下：

```php
function foo($model, $attribute) {
    // ... compute $value ...
    return $value;
}
```

> 补充：如何判断待测值是否为空，被写在另外一个话题的[处理空输入](input-validation.md#handling-empty-inputs)章节。


## [[yii\validators\NumberValidator|double（双精度浮点型）]] <a name="double"></a>

```php
[
    // 检查 "salary" 是否为浮点数
    ['salary', 'double'],
]
```

该验证器检查输入值是否为双精度浮点数。他等效于 [number](#number) 验证器。

- `max`：上限值（含界点）。若不设置，则验证器不检查上限。
- `min`：下限值（含界点）。若不设置，则验证器不检查下限。


## [[yii\validators\EmailValidator|email（电子邮件）]] <a name="email"></a>

```php
[
    // 检查 "email" 是否为有效的邮箱地址
    ['email', 'email'],
]
```

该验证器检查输入值是否为有效的邮箱地址。

- `allowName`：检查是否允许带名称的电子邮件地址 (e.g. `张三 <John.san@example.com>`)。 默认为 false。
- `checkDNS`：检查邮箱域名是否存在，且有没有对应的 A 或 MX 记录。不过要知道，有的时候该项检查可能会因为临时性 DNS 故障而失败，哪怕它其实是有效的。默认为 false。
- `enableIDN`：验证过程是否应该考虑 IDN（internationalized domain names，国际化域名，也称多语种域名，比如中文域名）。默认为 false。要注意但是为使用 IDN 验证功能，请先确保安装并开启 `intl` PHP 扩展，不然会导致抛出异常。


## [[yii\validators\ExistValidator|exist（存在）]] <a name="exist"></a>

```php
[
    // a1 需要在 "a1" 特性所代表的字段内存在
    ['a1', 'exist'],

    // a1 必需存在，但会用 a2 来检验其值的存在性
    ['a1', 'exist', 'targetAttribute' => 'a2'],

    // a1 和 a2 需要共同存在，且他们都会接收到错误信息
    [['a1', 'a2'], 'exist', 'targetAttribute' => ['a1', 'a2']],

    // a1 和 a2 需要共同存在，只有 a1 会接收到错误信息
    ['a1', 'exist', 'targetAttribute' => ['a1', 'a2']],

    // a1 needs to exist by checking the existence of both a2 and a3 (using a1 value)
    ['a1', 'exist', 'targetAttribute' => ['a2', 'a1' => 'a3']],

    // a1 必需存在，若 a1 为数组，则其每个子元素都必须存在。
    ['a1', 'exist', 'allowArray' => true],
]
```

该验证器检查输入值是否在表字段中存在。它只对[活动记录](db-active-record.md)模型的特性起作用。它支持对一个或多过字段的验证。

- `targetClass`：the name of the [Active Record](db-active-record.md) class that should be used
  to look for the input value being validated. If not set, the class of the model currently being validated will be used.
- `targetAttribute`：the name of the attribute in `targetClass` that should be used to validate the existence
  of the input value. If not set, it will use the name of the attribute currently being validated.
  You may use an array to validate the existence of multiple columns at the same time. The array values
  are the attributes that will be used to validate the existence, while the array keys are the attributes
  whose values are to be validated. If the key and the value are the same, you can just specify the value.
- `filter`：additional filter to be applied to the DB query used to check the existence of the input value.
  This can be a string or an array representing the additional query condition (refer to [[yii\db\Query::where()]]
  on the format of query condition), or an anonymous function with the signature `function ($query)`, where `$query`
  is the [[yii\db\Query|Query]] object that you can modify in the function.
- `allowArray`：whether to allow the input value to be an array. 默认为 false. If this property is true
  and the input is an array, then every element of the array must exist in the target column. Note that
  this property cannot be set true if you are validating against multiple columns by setting `targetAttribute` as an array.


## [[yii\validators\FileValidator|file（文件）]] <a name="file"></a>

```php
[
    // 检查 "primaryImage" 是否为 PNG, JPG 或 GIF 格式的上传图片。
    // 文件大小必须小于  1MB
    ['primaryImage', 'file', 'extensions' => ['png', 'jpg', 'gif'], 'maxSize' => 1024*1024*1024],
]
```

该验证器检查输入值是否为一个有效的上传文件。

- `extensions`：可接受上传的文件扩展名列表。它可以是数组，也可以是用空格或逗号分隔各个扩展名的字符串 (e.g. "gif, jpg")。
  扩展名大小写不敏感。默认为 null，意味着所有扩展名都被接受。
- `mimeTypes`：可接受上传的 MIME 类型列表。它可以是数组，也可以是用空格或逗号分隔各个 MIME 的字符串 (e.g. "image/jpeg, image/png")。
  Mime 类型名是大小写不敏感的。默认为 null，意味着所有 MIME 类型都被接受。
- `minSize`：上传文件所需最少多少 Byte 的大小。默认为 null，代表没有下限。
- `maxSize`：上传文件所需最多多少 Byte 的大小。默认为 null，代表没有上限。
- `maxFiles`：给定特性最多能承载多少个文件。默认为 1，代表只允许单文件上传。若值大于一，那么输入值必须为包含最多 `maxFiles` 个上传文件元素的数组。
- `checkExtensionByMimeType`：是否通过文件的 MIME 类型来判断其文件扩展。若由 MIME 判定的文件扩展与给定文件的扩展不一样，则文件会被认为无效。默认为 true，代表执行上述检测。

`FileValidator` 通常与 [[yii\web\UploadedFile]] 共同使用。请参考 [文件上传](input-file-upload.md)章节来了解有关文件上传与上传文件的检验的全部内容。


## [[yii\validators\FilterValidator|filter（过滤器）]] <a name="filter"></a>

```php
[
    // trim "username" and "email" inputs
    [['username', 'email'], 'filter', 'filter' => 'trim', 'skipOnArray' => true],

    // normalize "phone" input
    ['phone', 'filter', 'filter' => function ($value) {
        // normalize phone input here
        return $value;
    }],
]
```

This validator does not validate data. Instead, it applies a filter on the input value and assigns it
back to the attribute being validated.

- `filter`：a PHP callback that defines a filter. This can be a global function name, an anonymous function, etc.
  The function signature must be `function ($value) { return $newValue; }`. This property must be set.
- `skipOnArray`：whether to skip the filter if the input value is an array. 默认为 false.
  Note that if the filter cannot handle array input, you should set this property to be true. Otherwise some
  PHP error might occur.

> Tip：If you want to trim input values, you may directly use [trim](#trim) validator.


## [[yii\validators\ImageValidator|image（图片）]] <a name="image"></a>

```php
[
    // 检查 "primaryImage" 是否为适当尺寸的有效图片
    ['primaryImage', 'image', 'extensions' => 'png, jpg',
        'minWidth' => 100, 'maxWidth' => 1000,
        'minHeight' => 100, 'maxHeight' => 1000,
    ],
]
```

该验证器检查输入值是否为代表有效的图片文件。它继承自 [file](#file) 验证器，并因此继承有其全部属性。除此之外，它还支持以下为图片检验而设的额外属性：

- `minWidth`：图片的最小宽度。默认为 null，代表无下限。
- `maxWidth`：图片的最大宽度。默认为 null，代表无上限。
- `minHeight`：图片的最小高度。 默认为 null，代表无下限。
- `maxHeight`：图片的最大高度。默认为 null，代表无上限。


## [[yii\validators\RangeValidator|in（范围）]] <a name="in"></a>

```php
[
    // 检查 "level" 是否为 1、2 或 3 中的一个
    ['level', 'in', 'range' => [1, 2, 3]],
]
```

该验证器检查输入值是否存在于给定列表的范围之中。

- `range`：用于检查输入值的给定值列表。
- `strict`：输入值与给定值直接的比较是否为严格模式（也就是类型与值都要相同，即全等）。默认为 false。
- `not`：是否对验证的结果取反。默认为 false。当该属性被设置为 true，验证器检查输入值是否**不在**给定列表内。
- `allowArray`：是否接受输入值为数组。当该值为 true 且输入值为数组时，数组内的每一个元素都必须在给定列表内存在，否则返回验证失败。


## [[yii\validators\NumberValidator|integer（整数）]] <a name="integer"></a>

```php
[
    // 检查 "age" 是否为整数
    ['age', 'integer'],
]
```

该验证器检查输入值是否为整形。

- `max`：上限值（含界点）。若不设置，则验证器不检查上限。
- `min`：下限值（含界点）。若不设置，则验证器不检查下限。


## [[yii\validators\RegularExpressionValidator|match（正则表达式）]] <a name="match"></a>

```php
[
    // 检查 "username" 是否由字母开头，且只包含单词字符
    ['username', 'match', 'pattern' => '/^[a-z]\w*$/i']
]
```

该验证器检查输入值是否匹配指定正则表达式。

- `pattern`：用于检测输入值的正则表达式。该属性是必须的，若不设置则会抛出异常。
- `not`：是否对验证的结果取反。默认为 false，代表输入值匹配正则表达式时验证成功。如果设为 true，则输入值不匹配正则时返回匹配成功。


## [[yii\validators\NumberValidator|number（数字）]] <a name="number"></a>

```php
[
    // 检查 "salary" 是否为数字
    ['salary', 'number'],
]
```

该验证器检查输入值是否为数字。他等效于 [double](#double) 验证器。

- `max`：上限值（含界点）。若不设置，则验证器不检查上限。
- `min`：下限值（含界点）。若不设置，则验证器不检查下限。


## [[yii\validators\RequiredValidator|required（必填）]] <a name="required"></a>

```php
[
    // 检查 "username" 与 "password" 是否为空
    [['username', 'password'], 'required'],
]
```

该验证器检查输入值是否为空，还是已经提供了。

- `requiredValue`：所期望的输入值。若没设置，意味着输入不能为空。
- `strict`：检查输入值时是否检查类型。默认为 false。当没有设置 `requiredValue` 属性时，若该属性为 true，验证器会检查输入值是否严格为 null；若该属性设为 false，该验证器会用一个更加宽松的规则检验输入值是否为空。

当设置了 `requiredValue` 属性时，若该属性为 true，输入值与 `requiredValue` 的比对会同时检查数据类型。

> 补充：如何判断待测值是否为空，被写在另外一个话题的[处理空输入](input-validation.md#handling-empty-inputs)章节。


## [[yii\validators\SafeValidator|safe（安全）]] <a name="safe"></a>

```php
[
    // 标记 "description" 为安全特性
    ['description', 'safe'],
]
```

该验证器并不进行数据验证。而是把一个特性标记为[安全特性](structure-models.md#safe-attributes)。


## [[yii\validators\StringValidator|string（字符串）]] <a name="string"></a>

```php
[
    // 检查 "username" 是否为长度 4 到 24 之间的字符串
    ['username', 'string', 'length' => [4, 24]],
]
```

该验证器检查输入值是否为特定长度的字符串。并检查特性的值是否为某个特定长度。

- `length`：指定待测输入字符串的长度限制。该属性可以被指定为以下格式之一：
     * 证书：the exact length that the string should be of;
     * 单元素数组：代表输入字符串的最小长度 (e.g. `[8]`)。这会重写 `min` 属性。
     * 包含两个元素的数组：代表输入字符串的最小和最大长度(e.g. `[8, 128]`)。
     这会同时重写 `min` 和 `max` 属性。
- `min`：输入字符串的最小长度。若不设置，则代表不设下限。
- `max`：输入字符串的最大长度。若不设置，则代表不设上限。
- `encoding`：待测字符串的编码方式。若不设置，则使用应用自身的 [[yii\base\Application::charset|charset]] 属性值，该值默认为 `UTF-8`。


## [[yii\validators\FilterValidator|trim（译为修剪/裁边）]] <a name="trim"></a>

```php
[
    // trim 掉 "username" 和 "email" 两侧的多余空格
    [['username', 'email'], 'trim'],
]
```

该验证器并不进行数据验证。而是，trim 掉输入值两侧的多余空格。注意若该输入值为数组，那它会忽略掉该验证器。


## [[yii\validators\UniqueValidator|unique（唯一）]] <a name="unique"></a>

```php
[
    // a1 needs to be unique in the column represented by the "a1" attribute
    ['a1', 'unique'],

    // a1 needs to be unique, but column a2 will be used to check the uniqueness of the a1 value
    ['a1', 'unique', 'targetAttribute' => 'a2'],

    // a1 and a2 need to be unique together, and they both will receive error message
    [['a1', 'a2'], 'unique', 'targetAttribute' => ['a1', 'a2']],

    // a1 and a2 need to be unique together, only a1 will receive error message
    ['a1', 'unique', 'targetAttribute' => ['a1', 'a2']],

    // a1 needs to be unique by checking the uniqueness of both a2 and a3 (using a1 value)
    ['a1', 'unique', 'targetAttribute' => ['a2', 'a1' => 'a3']],
]
```

This validator checks if the input value is unique in a table column. It only works
with [Active Record](db-active-record.md) model attributes. It supports validation against
either a single column or multiple columns.

- `targetClass`：the name of the [Active Record](db-active-record.md) class that should be used
  to look for the input value being validated. If not set, the class of the model currently being validated will be used.
- `targetAttribute`：the name of the attribute in `targetClass` that should be used to validate the uniqueness
  of the input value. If not set, it will use the name of the attribute currently being validated.
  You may use an array to validate the uniqueness of multiple columns at the same time. The array values
  are the attributes that will be used to validate the uniqueness, while the array keys are the attributes
  whose values are to be validated. If the key and the value are the same, you can just specify the value.
- `filter`：additional filter to be applied to the DB query used to check the uniqueness of the input value.
  This can be a string or an array representing the additional query condition (refer to [[yii\db\Query::where()]]
  on the format of query condition), or an anonymous function with the signature `function ($query)`, where `$query`
  is the [[yii\db\Query|Query]] object that you can modify in the function.


## [[yii\validators\UrlValidator|url（网址）]] <a name="url"></a>

```php
[
    // 检查 "website" 是否为有效的 URL。若没有 URI 方案，则给 "website" 特性加 "http://" 前缀
    ['website', 'url', 'defaultScheme' => 'http'],
]
```

该验证器检查输入值是否为有效 URL。

- `validSchemes`：用于指定那些 URI 方案会被视为有效的数组。默认为 `['http', 'https']`，代表 `http` 和 `https` URLs 会被认为有效。
- `defaultScheme`：若输入值没有对应的方案前缀，会使用的默认 URI 方案前缀。默认为 null，代表不修改输入值本身。
- `enableIDN`：验证过程是否应该考虑 IDN（internationalized domain names，国际化域名，也称多语种域名，比如中文域名）。默认为 false。要注意但是为使用 IDN 验证功能，请先确保安装并开启 `intl` PHP 扩展，不然会导致抛出异常。